;;;
;;; zil-disasm.lisp
;;;

;;; instruction format
;;;
;;; from 
;;; http://www.ifarchive.org/if-archive/infocom/interpreters/specification/zmach06e.txt

#|
7.2.  Decoding an instruction

If (in V5+) the first byte of the instruction is $BE, it is an extended
instruction.  The second byte contains the EXT opcode.  Then follow the
operand types (one byte) and the operands themselves, in the same format as
for variable instructions (discussed below).  It is an error for an extended
instruction to occur in V1-4.

Otherwise, the instruction format depends on the top bits of the first byte:

o    %0: long;

o    %10: short;

o    %11: (double) variable.

The first byte of a long instruction is of the form %0abxxxxx.  Here, %xxxxx
is the 2OP opcode number, and %a and %b are abbreviated type-indicators for the
two operands:

o    %0: a byte constant;

o    %1: a variable number.

These correspond to the types %01 and %10, respectively.  The next two bytes of
the instruction contain the operands, first %a, then %b.

The first byte of a short instruction is of the form %10ttxxxx.  Here, %tt is
the type of the operand (or %11 if absent), and %xxxx is the 1OP (or if the
operand is absent:  0OP) opcode number.  If the operand is present, it follows
the first byte.

The first byte of a variable instruction is of the form %11axxxxx (except for
the two double variable instructions, discussed below), where %xxxxx is the
(2OP or VAR) opcode number.  If %a is %0 then this instruction contains a 2OP
opcode; if %1, a VAR opcode.  It is an error for a variable instruction with
a 2OP opcode not to have two operands, except when the opcode is 2OP:$1 (i.e.,
the instruction is JE).  The second byte is a type byte, and contains the
type information for the operands.  It is divided into pairs of two bits, each
indicating the type of an operand, beginning with the top bits.  The following
bytes contain the operands in that order.  A %11 pair means `no operand'; it is
an error if a %11 bit pair occurs before a non-%11 pair.

There are two double variable instructions, viz. %11101100 ($EC, opcode VAR:$C)
and %11111010 ($FA, opcode VAR:$1A).  Their structure is identical to that of
variable instructions, except that they do not have a type byte, but instead a
type word.

|#

(defun zil-instruction-decode (first-byte)

  (if (= first-byte #xBE)
    (list :extended-instruction)

    (let ((top-bit (logbitp 7 first-byte)))
      (labels ((long-op-type (bit)
                 (ecase bit
                   (0 :byte-constant)
                   (1 :variable-number)))

               (short-op-type (twobits)
                 (ecase twobits
                   (0 :word-constant)
                   (1 :byte-constant)
                   (2 :variable-number)
                   (3 :none)))

               (double-operand-type (bit)
                 (ecase bit
                   (0 :2op)
                   (1 :var)))
               
               (zil-long-instruction-decode ()
                 (let ((opcode (ldb (byte 5 0) first-byte))
                       (type-op1 (ldb (byte 1 6) first-byte))
                       (type-op2 (ldb (byte 1 5) first-byte)))
                   (list :2op opcode (long-op-type type-op1)
                         (long-op-type type-op2))))
               
               (zil-short-instruction-decode ()
                 (let ((op-type (ldb (byte 2 4) first-byte))
                       (opcode (ldb (byte 4 0) first-byte)))
                   (let ((op-sym (short-op-type op-type)))
                     (if (eql op-sym :none)
                       (list :0op opcode) 
                       (list :1op opcode op-sym)))))
               
               (zil-double-instruction-decode ()
                 (let ((op-type (ldb (byte 1 5) first-byte))
                       (opcode (ldb (byte 5 0) first-byte)))
                   (list (double-operand-type op-type) opcode))))
        
        (if top-bit
          (let ((top-two (ldb (byte 2 6) first-byte)))
            (ecase top-two
              (#b10 (zil-short-instruction-decode))
              (#b11 (zil-double-instruction-decode))))
          (zil-long-instruction-decode))))))

(defconstant +zil-0op-mnemonic-lookup+
  #("RTRUE" "RFALSE"
    "PRINT" "PRINT_RTRUE" ;; both have <string>
    "NOP" 
    "SAVE" ;; version 1--3 <branch> , version 4 <result>
    "RESTORE" ;; version 1--3 <branch> , version 4 <result>
    "RESTART" 
    "RET_PULLED"
    "POP"  ;; version 1--4 (version 5+: CATCH <result>)
    "QUIT"
    "NEW_LINE"
    "SHOW_STATUS"
    "VERIFY" ;; version 3+ <branch>
    "EXTENDED" ;; version 5+
    "PIRACY" ;; version 5+ <branch>
) "Mnemonic table for 0op opcodes")

(defconstant +zil-1op-mnemonic-lookup+
  #("JZ"              ;; a <branch>             ; opcode 0
    "GET_SIBLING"     ;; obj <result> <branch>  ; opcode 1
    "GET_CHILD"       ;; obj <result> <branch>
    "GET_PARENT"      ;; obj <result>          ; opcode 3
    "GET_PROP_LEN"    ;; baddr <result>
    "INC" "DEC"       ;; var             ; opcodes 5, 6
    "PRINT_ADDR"      ;; baddr
    "CALL_F0"         ;; raddr <result>  ; opcode 8
    "REMOVE_OBJ"      ;; obj
    "PRINT_OBJ"       ;; obj
    "RET"             ;; a
    "JUMP"            ;; s
    "PRINT_PADDR"     ;; saddr
    "LOAD"            ;; var <result>     ; opcode 14
    "NOT"             ;; version 1--4: a <result>, 5+: CALL_PO raddr ; opcode 15
    ) "Mnemonic table for 1op opcodes")

(defconstant +zil-2op-mnemonic-lookup+
  #("2OP:$0"        ;; ??
    "JE"            ;; a [b1 b2 b3] <branch>
    "JL"            ;; s t <branch>
    "JG"            ;; s t <branch>
    "DEC_JL"        ;; var s <branch>
    "INC_JG"        ;; var t <branch>
    "JIN"           ;; obj n <branch>
    "TEST"          ;; a b <branch>    ; opcode 7
    "OR"            ;; a b <result>
    "AND"           ;; a b <result>
    "TEST_ATTR"     ;; obj attr <branch> ; opcode 10
    "SET_ATTR"      ;; obj attr
    "CLEAR_ATTR"    ;; obj attr
    "STORE"         ;; var a
    "INSERT_OBJ"    ;; obj1 obj2
    "LOADW"         ;; baddr n <result>  ; opcode 15
    "LOADB"         ;; baddr n <result>
    "GET_PROP"      ;; obj prop <result>
    "GET_PROP_ADDR" ;; obj prop <result>
    "GET_NEXT_PROP" ;; obj prop <result>
    "ADD"           ;; a b <result>
    "SUB"           ;; a b <result>
    "MUL"           ;; a b <result>
    "DIV"           ;; a b <result>
    "MOD"           ;; a b <result>
    "CALL_F1"       ;; version 4+: raddr a1 <result> ; opcode 25
    "CALL_P1"       ;; version 5+: raddr a1
    "SET_COLOR"     ;; version 5+: f b 
    "THROW"         ;; version 5+: a fp ; opcode 28
    "2OP:$1D" 
    "2OP:$1E"
    "2OP:$1F"
    ) "Mnemonic table for 2op opcodes")

(defconstant +zil-var-mnemonic-lookup+
  #("CALL_FV" ;; raddr [a1 a2 a3] <result> ; opcode 0
    "STOREW"  ;; baddr n a
    "STOREB"  ;; baddr n byte
    "PUT_PROP" ;; obj prop a
    "READ"    ;; version 1--3: baddr1 baddr2, version 4: baddr1 baddr2 [time raddr]
              ;; version 5: baddr1 baddr2 [time raddr] <result>  ; opcode 4
    "PRINT_CHAR" ;; n
    "PRINT_NUM" ;; s
    "RANDOM" ;; s <result> ; opcode 7
    "PUSH"   ;; a
    "PULL" ;; version 1--5: var, version 6+: PULL [baddr] var
    "SPLIT_SCREEN" ;; version 3+: n 
    "SET_WINDOW" ;; 3+: n
    "CALL_FD"
    "ERASE_WINDOW"
    "ERASE_LINE"
    "SET_CURSOR"
    "GET_CURSOR"
    "SET_TEXT_STYLE"
    "BUFFER_MODE"
    "OUTPUT_STREAM" ;; version 3-4: s
    "INPUT_STREAM" ;; n
    "SOUND" ;; 3+ n [op time raddr]
    "READ_CHAR"
    "SCAN_TABLE"
    "NOT" 
    "CALL_PV"
    "CALL_PD"
    "TOKENIZE"
    "ENCODE_TEXT"
    "COPY_TABLE"
    "PRINT_TABLE"
    "CHECK_ARG_COUNT"
    ) "Mnemonics for var instructions.")

(defun disassemble-zil-instruction (code-array offset address abbreviation-table)
  (let* ((first-byte (aref code-array offset))
         (decoding (zil-instruction-decode first-byte))
         (cur-byte (1+ offset))) ;; pointer which is pushed along as more and more of the instruction 
                                ;; is decoded
    (labels ((gobble-bytes (n)
               (let ((val 0))
                 (dotimes (i n val)
                   (setf val (+ (* val 256) (aref code-array cur-byte)))
                   (incf cur-byte))))

             (gobble-variable-operand-bits (max-bytes &optional (previous-bits nil))
               (let ((byte (gobble-bytes 1))
                     (op-bit-list previous-bits))

                 (dotimes (i 4)
                   (let ((bitpair (ldb (byte 2 (- 6 (* i 2))) byte)))
                     (push bitpair op-bit-list)))

                 (if (and (= (car op-bit-list) 3) (> max-bytes 1))
                   (gobble-variable-operand-bits (1- max-bytes) op-bit-list)
                   (nreverse (remove-if #'(lambda (bits) (= bits 3)) op-bit-list)))))
      
             (gobble-variable-operand-types (&optional (max-bytes 1))
               (mapcar #'(lambda (bitpair)
                           (case bitpair
                             (0 :word-constant)
                             (1 :byte-constant)
                             (2 :variable-number)
                             (3 :none))) (gobble-variable-operand-bits max-bytes)))

                     
             (gobble-operand (op-type)
               (ecase op-type
                 (:word-constant (gobble-bytes 2))
                 (:byte-constant (gobble-bytes 1))
                 (:variable-number (format nil "var~D" (gobble-bytes 1)))
                 (:none nil)))
               
             (gobble-result ()
               (gobble-operand :variable-number))

             (gobble-branch-address ()
               (let* ((first-byte (gobble-bytes 1))
                      (sense-reversed (not (logbitp 7 first-byte)))
                      (is-single-byte (logbitp 6 first-byte))
                      (low-6-bits (ldb (byte 6 0) first-byte)))
                 (values (let ((branch-offset
                                (if (not is-single-byte)
                                  (let ((low-byte (gobble-bytes 1))
                                        (int 0))
                                    (setf (ldb (byte 8 0) int) low-byte
                                          (ldb (byte 6 8) int) low-6-bits)
                                    (if (> int 8191)
                                      (- int 8192)
                                      int))
                                  low-6-bits)))
                           (case branch-offset
                             (0 :rfalse)
                             (1 :rtrue)
                             (t (+ address (- cur-byte offset) -2 branch-offset))))
                         sense-reversed))))
                   
      (let ((instruction-string
             (ecase (car decoding)
               (:0op (let* ((opcode (second decoding))
                            (mnemonic (aref +zil-0op-mnemonic-lookup+ opcode)))
                       (case opcode
                         ((2 3) (multiple-value-bind (zstring word-count)
                                                     (zil-decode-string code-array cur-byte
                                                                        abbreviation-table)
                                  (incf cur-byte (* 2 word-count))
                                  (format nil "~A ~S" mnemonic zstring)))
                                  
                         ;;; in version 4: 0OP:5, 0OP6 have <result> not <branch>
                         
                         ((5 6 14 15) (multiple-value-bind (branch-destination sense-reversed)
                                                           (gobble-branch-address)
                                          (format nil "~A ~@[~~~]~A" mnemonic sense-reversed 
                                                  branch-destination)))
                         (t (format nil "~A" mnemonic)))))

               (:1op (let* ((opcode (second decoding))
                            (mnemonic (aref +zil-1op-mnemonic-lookup+ opcode))
                            (opsym (third decoding))
                            (operand-value (gobble-operand opsym)))
                       (case opcode

                         (0 ; <branch> only
                          (multiple-value-bind (branch-destination sense-reversed)
                                               (gobble-branch-address)
                            (format nil "~A ~@[~A~] ~:[~;~~~]~A" mnemonic operand-value 
                                    sense-reversed branch-destination)))

                         ((3 4 8 14 15) ;; <result> only  (not 15 in version 5+!)
                          (format nil "~A ~@[~A~] -> ~A" mnemonic operand-value 
                                  (gobble-result)))

                         ((1 2) ; <result> <branch> both.
                          (multiple-value-bind (branch-destination sense-reversed)
                                               (gobble-branch-address)
                            (format nil "~A ~@[~A~] ~:[~;~~~]~A -> ~A" mnemonic operand-value
                                    sense-reversed branch-destination 
                                    (gobble-result))))
                         
                         (t (format nil "~A ~@[~A~]" mnemonic operand-value)))))
                          
                         

               (:2op (let* ((opcode (second decoding))
                            (mnemonic (aref +zil-2op-mnemonic-lookup+ opcode))
                            (remaining (subseq decoding 2))
                            (operand-types (if (null remaining) ;; variable operands
                                             (gobble-variable-operand-types)
                                             remaining))
                            (operand-values (mapcar #'gobble-operand operand-types)))

                       (case opcode 

                         ((1 2 3 4 5 6 7 10)  ;; <branch>

                          (multiple-value-bind (branch-destination sense-reversed)
                                               (gobble-branch-address)
                            (format nil "~A~{ ~A~} ~:[~;~~~]~A" mnemonic operand-values
                                    sense-reversed branch-destination)))

                         ((8 9 15 16 17 18 19 20 21 22 23 24 25) ;; <result>
                          (format nil "~A~{ ~A~} -> ~A" mnemonic operand-values 
                                  (gobble-result)))

                         (t (format nil "~A~{ ~A~}" mnemonic operand-values)))))

               (:var (let* ((opcode (second decoding))
                            (mnemonic (aref +zil-var-mnemonic-lookup+ opcode))
                            (operand-types (gobble-variable-operand-types))
                            (operand-values (mapcar #'gobble-operand operand-types)))
                       
                       (case opcode
                         ((0 7) ;; <result> , also opcode 4 if version5+
                          (format nil "~A~{ ~A~} -> ~A" mnemonic operand-values
                                  (gobble-result)))
                         
                         (t (format nil "~A~{ ~A~}" mnemonic operand-values))))))))

          
        (values (format nil "~6,'0D: ~A" address instruction-string)
                cur-byte)))))
                

(defun disasm-zil (code-array start-offset end-offset start-address abbreviation-table
                              &optional (output-stream t))
  (do ((pc start-address)
       (offset start-offset))
      ((>= offset end-offset) nil)
    (multiple-value-bind (asm-line next-offset)
          (disassemble-zil-instruction code-array offset pc abbreviation-table)
      
      (format output-stream "~A~%" asm-line)
      (setf pc (+ pc (- next-offset offset)))
      (setf offset next-offset))))
  
                
(defclass zil-char-state ()
  ((current-case :initarg :current-case :accessor current-case)
   (zscii-expansion-count :initarg :zscii-expansion-count :accessor zscii-expansion-count)
   (zscii-so-far :initarg :zscii-so-far :accessor zscii-so-far)
   (expanding-abbreviation :initarg :expanding-abbreviation :accessor expanding-abbreviation)))

(defun new-zil-char-state ()
  (make-instance 'zil-char-state 
    :current-case :lowercase
    :zscii-expansion-count nil
    :zscii-so-far 0
    :expanding-abbreviation nil))
       
(defun zscii-char (code)
  "Returns a Zscii character. This is a ten-bit character code."
  (code-char code))


(defmethod zil-next-char-state ((current-state zil-char-state) zcode abbreviation-table)

  "Returns two values: 
     First: a character or character string to be added to the string, or NIL if there
            are no characters to add
     Second: a mutated current-state representing the next state of the decoding process."

  (labels ((expand-zil-abbreviation (abbrev-number)
             (aref abbreviation-table abbrev-number)))
    
    (with-slots (current-case zscii-expansion-count
                              zscii-so-far expanding-abbreviation) current-state
      
      (when (and zscii-expansion-count expanding-abbreviation)
        (error "Invalid zil-char-state."))

      (unless (and (numberp zcode) (<= 0 zcode 31))
        (error "Illegal zcode ~A" zcode))
      
      (cond 
       ((and (numberp zscii-expansion-count)
             (= zscii-expansion-count 0)) (progn (setf zscii-so-far (* 32 zcode))
                                                 (setf zscii-expansion-count 1)
                                                 (values nil current-state)))
       
       ((and (numberp zscii-expansion-count)
             (= zscii-expansion-count 1))  (let ((zscii-code (+ zscii-so-far zcode)))
                                             (setf zscii-so-far 0
                                                   zscii-expansion-count nil
                                                   current-case :lowercase)
                                             (values (zscii-char zscii-code) current-state)))
       
       (expanding-abbreviation (let ((expansion (expand-zil-abbreviation (+ (* 32 (1- expanding-abbreviation))
                                                                           zcode))))
                                 (setf expanding-abbreviation nil)
                                 (values expansion current-state)))
       
       ((zerop zcode) (progn
                        (setf current-case :lowercase)
                        (values #\Space current-state)))
       
       ((<= 1 zcode 3) (progn
                         (setf expanding-abbreviation zcode)
                         (values nil current-state)))
       
       
       ((= zcode 4) (progn
                      (setf current-case (ecase current-case
                                           (:lowercase :uppercase)
                                           (:uppercase :punctuation)
                                           (:punctuation :lowercase)))
                      (values nil current-state)))
       
       ((= zcode 5) (progn 
                      (setf current-case (ecase current-case
                                           (:lowercase :punctuation)
                                           (:uppercase :lowercase)
                                           (:punctuation :uppercase)))
                      (values nil current-state)))
       
       ((eql current-case :punctuation) (progn 
                                          (setf current-case :lowercase)
                                          (case zcode
                                            (6 (progn
                                                 (setf zscii-expansion-count 0
                                                       zscii-so-far 0)
                                                 (values nil current-state)))
                                            (7 (values #\Newline current-state))
                                            (t (values (elt "0123456789.,!?_#'\"/\\-:()" (- zcode 8))
                                                       current-state)))))
       
       ((eql current-case :lowercase) (values (elt "abcdefghijklmnopqrstuvwxyz" (- zcode 6))
                                              current-state))
       
       ((eql current-case :uppercase) (progn 
                                        (setf current-case :lowercase)
                                        (values (elt "ABCDEFGHIJKLMNOPQRSTUVWXYZ" (- zcode 6))
                                                current-state)))
       
       (t (error "Unknown zil-char-state."))))))
  
#|     
  get slots of current state 

  if zscii-expansion-count is 0
     set zscii-so-far to zcode
     set zscii-expansion-count to 1
     return NIL for the character
  
  if zscii-expansion-count is 1
     append zcode to zscii-so-far, shifted to give a 10-bit result.
     translate zscii-so-far to a character
     return it, and clear out zscii-expansion, and case to lowercase

  if expanding-abbreviation is non-nil
     that is, in range 1..3
     then fetch the abbreviation (+ (* 32 (- expanding-abbreviation 1)) zcode)
         that is the string to add. (do we need to fix the case of the first character we 
         are in uppercase? what about punctuation? What is the point of having abbreviation
         codes in the shifted character sets???
         what is the case to return in the next state?

  OK. 
  0: return space, lowercase is next case
  1..3: return NIL, expanding abbreviation=zcode is next state
  4: if lowercase, uppercase is next case
     if uppercase, punct is next case 
     if punct, lowercase is next case
     for all of these, NIL character
  5: if lowercase, punct is next
     if uppercase, lowercase is next
     if punct, uppercase is next
  
  6: if punctutation, zscii-expansion <- 0, NIL character
  7: if punctuation NEWLINE, lowercase is next case.
  8..31: lookup punctuation character, lowercase is next case

  6..31: if lowercase, uppercase: lookup character, lowercase is next case        
|#     
  

(defun zil-code-array (byte-array offset word-count)
  (let* ((ch-count (* word-count 3))
         (code-array (make-array ch-count :element-type '(integer 0 31)
                                 :initial-element 0)))
    (dotimes (w word-count code-array)
      (let* ((b (+ offset (* w 2)))
             (c (* w 3))
             (w (+ (* 256 (aref byte-array b)) (aref byte-array (1+ b)))))
        (setf (aref code-array c) (ldb (byte 5 10) w)
              (aref code-array (1+ c)) (ldb (byte 5 5) w)
              (aref code-array (+ c 2)) (ldb (byte 5 0) w))))))

(defun zil-decode-string (byte-array offset abbreviation-table)
  (let* ((word-count (do ((found nil)
                          (pos offset (+ pos 2)))
                         (found (/ (- pos offset) 2))
                       (setf found (logbitp 7 (aref byte-array pos)))))
         (ch-count (* word-count 3))
         (zarray (zil-code-array byte-array offset word-count))
         (s (make-array  ch-count :fill-pointer 0 :element-type 'character
                        :adjustable t))
         (state (new-zil-char-state)))
    
    (dotimes (c ch-count (values s word-count))
      (multiple-value-bind (result next-state)
                           (zil-next-char-state state (aref zarray c) abbreviation-table)
        (setf state next-state)
        (typecase result
          (character (vector-push-extend result s))
          (string (dotimes (sc (length result))
                    (vector-push-extend (aref result sc) s)))
          (null)
          (t (error "Bad result from zil-next-char-state")))))))

(defconstant +zil-abbreviation-table-length+ (* 3 32) 
  "Number of abbreviations in an abbreviation table.")

(defun make-zil-abbreviation-table (byte-array pointer-table-offset)

  "Given a POINTER-TABLE-OFFSET reads a series of word pointers from BYTE-ARRAY,
which is assumed to start at word 0 of the ZIL program image, then reads 
the abbreviations (Z-strings) stored at each word address pointed to,
and stores them in an array, which is returned."

  (let ((abbrev-table (make-array +zil-abbreviation-table-length+ 
                                  :element-type 'string :initial-element "")))

    (dotimes (w +zil-abbreviation-table-length+ abbrev-table)
      (let* ((pointer-address (+ pointer-table-offset (* 2 w)))
             (pointer (+ (* 256 (aref byte-array pointer-address))
                         (aref byte-array (1+ pointer-address))))
             (byte-offset (* 2 pointer))
             (abbrev (zil-decode-string byte-array byte-offset abbrev-table)))
        (setf (aref abbrev-table w) abbrev)))))

#|


The object table is held in dynamic memory and its byte address is stored in the word at $0a in the header.
(Recall that objects have flags attached called attributes, numbered from 0 upward, and variables
attached called properties, numbered from 1 upward. An object need not provide every property.) 


12.2

The table begins with a block known as the property defaults table. This contains 31 words in
Versions 1 to 3 and 63 in Versions 4 and later. When the game attempts to read the value of
property n for an object which does not provide property n, the n-th entry in this
table is the resulting value. 


12.3

Next is the object tree. Objects are numbered consecutively from 1 upward, with object
number 0 being used to mean "nothing" (though there is formally
no such object). The table consists of a list of entries, one for each object. 


12.3.1

In Versions 1 to 3, there are at most 255 objects, each having a 9-byte entry as follows: 

   the 32 attribute flags     parent     sibling     child   properties
   ---32 bits in 4 bytes---   ---3 bytes------------------  ---2 bytes--

parent, sibling and child must all hold valid object numbers. The properties 
pointer is the byte address of the list of properties attached to the object.
Attributes 0 to 31 are flags (at any given time, they are either on (1) or off (0))
and are stored topmost bit first: e.g., attribute 0 is stored in bit 7 of the first
byte, attribute 31 is stored in bit 0 of the fourth. 


12.3.2

In Version 4 and later, there are at most 65535 objects, each having a 14-byte entry as follows: 

    the 48 attribute flags     parent    sibling   child     properties
   ---48 bits in 6 bytes---   ---3 words, i.e. 6 bytes----  ---2 bytes--


12.4

Each object has its own property table. Each of these can be anywhere in dynamic memory
(indeed, a game can legally change an object's properties table address in play, provided
the new address points to another valid properties table). 
The header of a property table is as follows: 

   text-length     text of short name of object
  -----byte----   --some even number of bytes---

where the text-length is the number of 2-byte words making up the text, which is stored
in the usual format. (This means that an object's short name is
limited to 765 Z-characters.) After the header, the properties are listed in descending
numerical order. (This order is essential and is not a matter of
convention.)


12.4.1

In Versions 1 to 3, each property is stored as a block 

   size byte     the actual property data
                ---between 1 and 8 bytes--

where the size byte is arranged as 32 times the number of data bytes minus one,
plus the property number. A property list is terminated by a size byte of
0. (It is otherwise illegal for a size byte to be a multiple of 32.) 


12.4.2

In Versions 4 and later, a property block instead has the form 

   size and number       the actual property data
  --1 or 2 bytes---     --between 1 and 64 bytes--

The property number occupies the bottom 6 bits of the first size byte. 


12.4.2.1

If the top bit (bit 7) of the first size byte is set, then there are two
size-and-number bytes as follows. In the first byte, bits 0 to 5 contain the property
number; bit 6 is undetermined (it is clear in all Infocom or Inform story files);
bit 7 is set. In the second byte, bits 0 to 5 contain the property data length,
counting in bytes; bit 6 is undetermined (it is set in Infocom story files,
but clear in Inform ones); bit 7 is always set. 


12.4.2.1.1

*** A value of 0 as property data length (in the second byte) should be 
interpreted as a length of 64. (Inform can compile such properties.) 


12.4.2.2

If the top bit (bit 7) of the first size byte is clear, then there is only
one size-and-number byte. Bits 0 to 5 contain the property number; bit 6 is either clear to
indicate a property data length of 1, or set to indicate a length of 2; bit 7 is clear. 
|#

(defun zil-read-9byte-property (byte-array offset)
  (let ((attr-flags 0)
        (parent (aref byte-array (+ offset 4)))
        (sibling (aref byte-array (+ offset 5)))
        (child (aref byte-array (+ offset 6)))
        (property-address 0))
    (dotimes (i 4)
      (setf (ldb (byte 8 (- 24 (* 8 i))) attr-flags)
            (aref byte-array (+ offset i))))
    (dotimes (i 2)
      (setf (ldb (byte 8 (- 8 (* 8 i))) property-address)
            (aref byte-array (+ offset 7 i))))
    
    (values attr-flags
            parent
            sibling
            child
            property-address)))

(defun zil-read-property-header (byte-array property-address abbreviation-table)
  (let ((word-len (aref byte-array property-address)))
    (if (zerop word-len)
      (values "" 0 0)
      (multiple-value-bind (name name-word-len)
                           (zil-decode-string byte-array (1+ property-address) abbreviation-table)
        (values name name-word-len word-len)))))

(defun zil-read-property-list (byte-array property-address abbreviation-table)
  (multiple-value-bind (name name-word-len word-len)
       (zil-read-property-header byte-array property-address abbreviation-table)
    (unless (= name-word-len word-len)
      (error "Property list has invalid header."))

    (let ((plist-start (+ property-address 1 (* 2 name-word-len))))
      (do ((paddr plist-start)
           (plist nil)
           (last-p nil))
          (last-p (cons (cons :name name) plist))
        (let ((size-byte (aref byte-array paddr)))
          (if (zerop size-byte)
            (setf last-p t) 
            (multiple-value-bind (size-1 propnum)
                                 (floor size-byte 32)
              (let* ((size (1+ size-1))
                     (data (subseq byte-array (1+ paddr) (+ 1 paddr size))))
                (push (cons propnum data) plist)
                (incf paddr (+ 1 size))))))))))

(defun zil-read-properties (byte-array offset abbreviation-table)
  (multiple-value-bind (attr-flags parent sibling child property-address)
                       (zil-read-9byte-property byte-array offset)
    (let ((property-list (zil-read-property-list byte-array property-address abbreviation-table)))
      (cons (cons :flags attr-flags)
            (cons (cons :parent parent)
                  (cons (cons :sibling sibling)
                        (cons (cons :child child)
                              property-list)))))))

(defun zil-object (byte-array property-table-start object-num abbreviation-table)
  (when (or (minusp object-num) (> object-num 255))
    (error "invalid object number ~D" object-num))
  (if (zerop object-num)
    nil
    (zil-read-properties byte-array (+ property-table-start 62 (* 9 (1- object-num)))
                         abbreviation-table)))

                  

    
        


  
        
          
        

    
    
  
